<?php
require_once '../models/User.php';
require_once '../utils/Auth.php';
require_once '../utils/helpers.php';

class UserController {
    private $userModel;
    
    public function __construct() {
        $this->userModel = new User();
    }

    public function index() {
        Auth::checkPermission(['admin']);
        
        $users = $this->userModel->findAll();
        include '../views/users/index.php';
    }

    public function create() {
        Auth::checkPermission(['admin']);
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleCreate();
        } else {
            include '../views/users/create.php';
        }
    }
    
    private function handleCreate() {
        if (!Auth::validateCsrfToken($_POST['csrf_token'])) {
            die('Token CSRF invalide');
        }

        $data = [
            'username' => sanitize($_POST['username']),
            'email' => sanitize($_POST['email']),
            'password' => $_POST['password'],
            'user_roles' => sanitize($_POST['user_roles'])
        ];

        // Validation
        $errors = [];
        
        if (empty($data['username'])) {
            $errors[] = "Le nom d'utilisateur est requis";
        }
        
        if (!validateEmail($data['email'])) {
            $errors[] = "L'email n'est pas valide";
        }
        
        if (strlen($data['password']) < 8) {
            $errors[] = "Le mot de passe doit contenir au moins 8 caractères";
        }
        
        if (!in_array($data['user_roles'], ['admin', 'chef_comptable', 'comptable'])) {
            $errors[] = "Rôle utilisateur invalide";
        }

        if (empty($errors)) {
            if ($this->userModel->create($data)) {
                $_SESSION['flash'] = ['type' => 'success', 'message' => 'Utilisateur créé avec succès'];
                redirect('/users');
            } else {
                $errors[] = "Erreur lors de la création de l'utilisateur";
            }
        }

        if (!empty($errors)) {
            $_SESSION['flash'] = ['type' => 'error', 'messages' => $errors];
            include '../views/users/create.php';
        }
    }

    public function edit($id) {
        Auth::checkPermission(['admin']);
        
        $user = $this->userModel->findById($id);
        
        if (!$user) {
            $_SESSION['flash'] = ['type' => 'error', 'message' => 'Utilisateur non trouvé'];
            redirect('/users');
        }
        
        include '../views/users/edit.php';
    }
    
    public function update($id) {
        Auth::checkPermission(['admin']);
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleUpdate($id);
        } else {
            redirect('/users');
        }
    }
    
    private function handleUpdate($id) {
        if (!Auth::validateCsrfToken($_POST['csrf_token'])) {
            die('Token CSRF invalide');
        }

        $data = [
            'username' => sanitize($_POST['username']),
            'email' => sanitize($_POST['email']),
            'user_roles' => sanitize($_POST['user_roles'])
        ];

        // Ajouter le mot de passe seulement si fourni
        if (!empty($_POST['password'])) {
            $data['password'] = $_POST['password'];
        }

        // Validation
        $errors = [];
        
        if (empty($data['username'])) {
            $errors[] = "Le nom d'utilisateur est requis";
        }
        
        if (!validateEmail($data['email'])) {
            $errors[] = "L'email n'est pas valide";
        }
        
        if (isset($data['password']) && strlen($data['password']) < 8) {
            $errors[] = "Le mot de passe doit contenir au moins 8 caractères";
        }
        
        if (!in_array($data['user_roles'], ['admin', 'chef_comptable', 'comptable'])) {
            $errors[] = "Rôle utilisateur invalide";
        }

        if (empty($errors)) {
            if ($this->userModel->update($id, $data)) {
                $_SESSION['flash'] = ['type' => 'success', 'message' => 'Utilisateur mis à jour avec succès'];
                redirect('/users');
            } else {
                $errors[] = "Erreur lors de la mise à jour de l'utilisateur";
            }
        }

        if (!empty($errors)) {
            $_SESSION['flash'] = ['type' => 'error', 'messages' => $errors];
            $user = $this->userModel->findById($id);
            include '../views/users/edit.php';
        }
    }

    public function delete($id) {
        Auth::checkPermission(['admin']);
        
        if (!Auth::validateCsrfToken($_GET['csrf_token'] ?? '')) {
            die('Token CSRF invalide');
        }

        // Empêcher l'auto-suppression
        if ($_SESSION['user']['id'] == $id) {
            $_SESSION['flash'] = ['type' => 'error', 'message' => 'Vous ne pouvez pas supprimer votre propre compte'];
            redirect('/users');
        }

        if ($this->userModel->delete($id)) {
            $_SESSION['flash'] = ['type' => 'success', 'message' => 'Utilisateur supprimé avec succès'];
        } else {
            $_SESSION['flash'] = ['type' => 'error', 'message' => 'Erreur lors de la suppression'];
        }
        
        redirect('/users');
    }

    public function profile() {
        Auth::checkPermission(['admin', 'chef_comptable', 'comptable']);
        
        $id = $_SESSION['user']['id'];
        $user = $this->userModel->findById($id);
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleProfileUpdate($id, $user);
        } else {
            include '../views/users/profile.php';
        }
    }
    
    private function handleProfileUpdate($id, $currentUser) {
        if (!Auth::validateCsrfToken($_POST['csrf_token'])) {
            die('Token CSRF invalide');
        }

        $data = [
            'email' => sanitize($_POST['email'])
        ];

        // Changement de mot de passe seulement si les trois champs sont remplis
        if (!empty($_POST['current_password']) && !empty($_POST['new_password']) && !empty($_POST['confirm_password'])) {
            // Vérifier le mot de passe actuel
            if (!password_verify($_POST['current_password'], $currentUser['password'])) {
                $_SESSION['flash'] = ['type' => 'error', 'message' => 'Mot de passe actuel incorrect'];
                include '../views/users/profile.php';
                return;
            }
            
            if ($_POST['new_password'] !== $_POST['confirm_password']) {
                $_SESSION['flash'] = ['type' => 'error', 'message' => 'Les nouveaux mots de passe ne correspondent pas'];
                include '../views/users/profile.php';
                return;
            }
            
            if (strlen($_POST['new_password']) < 8) {
                $_SESSION['flash'] = ['type' => 'error', 'message' => 'Le nouveau mot de passe doit contenir au moins 8 caractères'];
                include '../views/users/profile.php';
                return;
            }
            
            $data['password'] = $_POST['new_password'];
        }

        if ($this->userModel->update($id, $data)) {
            $_SESSION['flash'] = ['type' => 'success', 'message' => 'Profil mis à jour avec succès'];
            
            // Mettre à jour l'email dans la session si changé
            if ($data['email'] !== $currentUser['email']) {
                $_SESSION['user']['email'] = $data['email'];
            }
        } else {
            $_SESSION['flash'] = ['type' => 'error', 'message' => 'Erreur lors de la mise à jour du profil'];
        }
        
        redirect('/profile');
    }
}
?>